using System;
using System.Data;
using System.Data.SqlClient;
using MethodBase = System.Reflection.MethodBase;
using ORDEREDCOMPONENT = gov.va.med.vbecs.Common.VbecsTables.OrderedComponent;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.ExceptionManagement;

namespace gov.va.med.vbecs.DAL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/23/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///	Data access component implementing database read/write functionality
		///	for OrderedComponent data. 
		///</summary>

		#endregion

	
	public class OrderedComponent
	{

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2726"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Datatable containing the list of component tasks</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="2727"> 
		///		<ExpectedInput>Invalid Group Number</ExpectedInput>
		///		<ExpectedOutput>Empty Table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets list of pending component tasks for a given division
		/// </summary>
		/// <returns>DataTable containing a list of pending component tasks</returns>
		public static DataTable GetComponentTaskList(string orderGroupNumber)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetComponentTaskList.divisioncode, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetComponentTaskList.ordergroupnumber, System.Data.SqlDbType.VarChar)
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			if (orderGroupNumber != "")
				prms[1].Value = orderGroupNumber;
			else
				prms[1].Value = DBNull.Value;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetComponentTaskList.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2846"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable containing all valid order for the respective division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2847"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>Retrieves component orders for a given division</summary>
		///<returns>DataTable containing component orders data</returns>
		public static DataTable GetComponentOrders()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientComponentOrdersAll.divisioncode, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientComponentOrdersAll.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4169"> 
		///		<ExpectedInput>PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataSet of OrderedUnit and OrderedComponent Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4170"> 
		///		<ExpectedInput>Empty PatientGuid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetRecentComponentOrders
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataSet GetRecentComponentOrders(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.VbecsTables.Patient.PatientGuid).ResString);
			}
			
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetRecentComponentOrdersForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetRecentComponentOrdersForPatient.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetRecentComponentOrdersForPatient.StoredProcName, prms);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8195"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Count of the recent component orders</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8196"> 
		///		<ExpectedInput>Invalid patient Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get a count of the recent component orders for a patient
		/// </summary>
		/// <param name="patientGuid">Patient Guid</param>
		/// <returns>Count of the recent component orders</returns>
		public static int GetRecentComponentOrdersCount(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.VbecsTables.Patient.PatientGuid).ResString);
			}
			
			int count = 0;

			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetRecentComponentOrdersForPatientCount.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetRecentComponentOrdersForPatientCount.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetRecentComponentOrdersForPatientCount.StoredProcName, prms);
			foreach(DataTable dt in ds.Tables)
			{
				foreach(DataRow dr in dt.Rows)
				{
					count += (int)dr["Total"];
				}
			}
			//
			return count;
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2848"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2849"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>Gets component classes that can be ordered in a given division</summary>
		///<returns>DataTable containing orderable component classes data</returns>
		public static DataTable GetOrderableComponentClasses()
		{
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderableComponentClasses.StoredProcName);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8042"> 
		///		<ExpectedInput>Valid ComponentClassId</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8043"> 
		///		<ExpectedInput>Invalid ComponentClassId</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean telling if any ordered components is are pending
		/// </summary>
		/// <param name="componentClassID"></param>
		/// <returns></returns>
		public static bool IsOrderedComponentPending(int componentClassID)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.IsOrderedComponentPending.divisioncode, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.IsOrderedComponentPending.componentclassid, System.Data.SqlDbType.Int)
				};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			if (componentClassID>0)
			{
				prms[1].Value = componentClassID;
			}
			else
			{
				prms[1].Value = DBNull.Value;
			}

			return new Common.StoredProcedure().GetBoolReturnValue( STOREDPROC.IsOrderedComponentPending.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2850"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2851"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// Saves ordered component data changes
		/// </summary>
		/// <param name="dtOrderComponent"></param>
		/// <param name="dtAcceptOrder"></param>
		/// <param name="dtWorkloadEvents"></param>
		/// <returns></returns>
		public static bool UpdateStatus(DataTable dtOrderComponent, DataTable dtAcceptOrder, DataTable dtWorkloadEvents)
		{	
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			
			sprocArray.Add(STOREDPROC.UpdateOrderedComponentStatus.StoredProcName);
			dtArray.Add(dtOrderComponent);
			sprocArray.Add(STOREDPROC.InsertExceptionEmergencyIssueOrder.StoredProcName);
			dtArray.Add(dtAcceptOrder);
			if(dtWorkloadEvents != null)
			{
				sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				dtArray.Add(dtWorkloadEvents);
			}

			int returnCode = (new Common.StoredProcedure()).TransactionalGetValue(sprocArray, dtArray);
			
			//check the final status of return code
			return(returnCode == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1196"> 
		///		<ExpectedInput>CPRS OrderNumber</ExpectedInput>
		///		<ExpectedOutput>DataSet of Order Data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1197"> 
		///		<ExpectedInput>Invalid CPRS Number</ExpectedInput>
		///		<ExpectedOutput>Empty DataSet</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets order data for CPRS update
		/// </summary>
		/// <param name="cprsOrderNumber"></param>
		/// <returns></returns>
		public static DataSet GetOrderData(string cprsOrderNumber)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderData.cprsordernumber, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetOrderData.divisioncode, System.Data.SqlDbType.VarChar),
			};

			prms[0].Value = cprsOrderNumber;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetOrderData.StoredProcName, prms);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1200"> 
		///		<ExpectedInput>OrderedComponentGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of TestResults</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1201"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static function to retrieve a list of blood unit test results for a given ordered component (Repeat XMs only)
		/// </summary>
		/// <param name="orderedComponentGuid">Ordered component unique identifier</param>
		/// <returns>Data table of blood unit test information</returns>
		public static DataTable GetBloodUnitTestResultsForOrderedComponent(Guid orderedComponentGuid)
		{
			if (orderedComponentGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.VbecsTables.OrderedComponent.OrderedComponentGuid).ResString);
			}
			
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedComponent.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedComponent.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = orderedComponentGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestResultsForOrderedComponent.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1206"> 
		///		<ExpectedInput>OrderedComponentGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1207"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets OrderedComponent data for Guid
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		public static DataTable GetOrderedComponent(System.Guid orderedComponentGuid)
		{
			if (orderedComponentGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.VbecsTables.OrderedComponent.OrderedComponentGuid).ResString);
			}
			 
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetOrderedComponent.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedComponent.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = orderedComponentGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedComponent.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4463"> 
		///		<ExpectedInput>Valid ordered component and blood unit Guids</ExpectedInput>
		///		<ExpectedOutput>Data table containing blood unit test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4464"> 
		///		<ExpectedInput>Empty orderedComponentGuid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of blood unit test results for a given ordered component (Repeat XMs only)
		/// </summary>
		/// <param name="orderedComponentGuid">Ordered component unique identifier</param>
		/// <param name="bloodUnitGuid">Blood unit unique identifier</param>
		/// <returns>Data table of blood unit test information</returns>
		public static DataTable GetBloodUnitTestResultsForOrderedComponent(Guid orderedComponentGuid, Guid bloodUnitGuid)
		{
			if (orderedComponentGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.VbecsTables.OrderedComponent.OrderedComponentGuid).ResString);
			}
			
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.VbecsTables.BloodUnit.BloodUnitGuid).ResString);
			}
			
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedComponent.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedComponent.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedComponent.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = orderedComponentGuid;
			prms[1].Value = bloodUnitGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode; 

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestResultsForOrderedComponent.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4289"> 
		///		<ExpectedInput>DivisionCode, Hours Needed, LastChecked time</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4290"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRequiredOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="hoursNeeded"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetRequiredOrders(string divisionCode, int hoursNeeded, System.DateTime lastChecked)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(ORDEREDCOMPONENT.DivisionCode).ResString);
			}
			
			if (lastChecked.Equals(new DateTime()))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("lastChecked").ResString);
			}
			
			if (hoursNeeded < 0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("hoursNeeded").ResString);
			}
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetNewOrderedComponentsForRequiredTimeFrame.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetNewOrderedComponentsForRequiredTimeFrame.lastchecked, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetNewOrderedComponentsForRequiredTimeFrame.requiredhours, System.Data.SqlDbType.Int)
			};
			
			prms[0].Value = divisionCode;
			prms[1].Value = lastChecked;
			prms[2].Value = hoursNeeded;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetNewOrderedComponentsForRequiredTimeFrame.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4291"> 
		///		<ExpectedInput>Division, DateTime</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4292"> 
		///		<ExpectedInput>Invalid DateTime</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetOrdersWithSpecimenInBloodBank
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetOrdersWithSpecimenInBloodBank(string divisionCode, System.DateTime lastChecked)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			if (lastChecked.Equals(new DateTime()))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("lastChecked").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetNewOrderedComponentsWithNoRequiredSpecimenOrSpecimenOnFile.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetNewOrderedComponentsWithNoRequiredSpecimenOrSpecimenOnFile.lastchecked, System.Data.SqlDbType.DateTime)
			};
			//
			prms[0].Value = divisionCode;
			prms[1].Value = lastChecked;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetNewOrderedComponentsWithNoRequiredSpecimenOrSpecimenOnFile.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4293"> 
		///		<ExpectedInput>Division, Time</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4294"> 
		///		<ExpectedInput>Invalid Time</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetNewOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetNewOrders(string divisionCode, System.DateTime lastChecked)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			if (lastChecked.Equals(new DateTime()))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("lastChecked").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetNewOrderedComponents.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetNewOrderedComponents.lastchecked, System.Data.SqlDbType.DateTime)
			};
			//
			prms[0].Value = divisionCode;
			prms[1].Value = lastChecked;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetNewOrderedComponents.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8250"> 
		///		<ExpectedInput>Ordered Component Guid</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8251"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the number of ordered units
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		/// <returns></returns>
		public static int GetOrderedComponentQuantity(System.Guid orderedComponentGuid)
		{
			if (orderedComponentGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.VbecsTables.OrderedComponent.OrderedComponentGuid).ResString);
			}
			 
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetOrderedComponent.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier)
			};

			prms[0].Value = orderedComponentGuid;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedComponentQuantity.StoredProcName, prms).Tables[0];
			return System.Convert.ToInt32(dt.Rows[0][Common.VbecsTables.OrderedComponent.RequiredUnitQuantity]);
		}
	}
}
